/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "themeiconlabel.h"
#include <QPainter>
#include <QGridLayout>
#include <QDebug>

ThemeIconLabel::ThemeIconLabel(QWidget *parent)
{
    this->setScaledContents(true);
    QHBoxLayout *hLyt = new QHBoxLayout(this);
    hLyt->setContentsMargins(0, 0, 0, 0);
    mPictrueLabel = new DrawIconLabel(mRadius - 2, this);
    hLyt->addWidget(mPictrueLabel);
}

ThemeIconLabel::ThemeIconLabel(const QPixmap &pixmap, QWidget *parent)
{
    this->setScaledContents(true);
    QHBoxLayout *hLyt = new QHBoxLayout(this);
    hLyt->setContentsMargins(0, 0, 0, 0);
    mPictrueLabel = new DrawIconLabel(pixmap, mRadius - 2, this);
    hLyt->addWidget(mPictrueLabel);
}

ThemeIconLabel::ThemeIconLabel(QStringList iStringList, QWidget *parent):
    mIconList(iStringList),
    QLabel(parent)
{
    QGridLayout *gLyt = new QGridLayout(this);
    gLyt->setContentsMargins(0, 0, 0, 0);
    gLyt->setSpacing(10);
    for (int i = 0; i < iStringList.count(); i++) {
        QLabel * label = new QLabel(this);
        label->setFixedSize(32, 32);
        label->setPixmap(pixmapToRound(iStringList.at(i), QSize(32, 32)));
        gLyt->addWidget(label, i / 3 , i % 3);
    }
}

ThemeIconLabel::ThemeIconLabel(QList<QPixmap> iconList, QWidget *parent):
    QLabel(parent)
{
    QGridLayout *gLyt = new QGridLayout(this);
    gLyt->setContentsMargins(0, 0, 0, 0);
    gLyt->setSpacing(10);
    for (int i = 0; i < iconList.count(); i++) {
        QLabel * label = new QLabel(this);
        label->setFixedSize(24, 24);
        label->setPixmap(iconList.at(i));
//        label->setPixmap(pixmapToRound(iconList.at(i), QSize(24, 24)));
        gLyt->addWidget(label, i / 3 , i % 3);
    }
}

QPixmap ThemeIconLabel::pixmapToRound(const QString &filePath, const QSize &scaledSize)
{
    QPixmap pixmap(filePath);

    qreal dpi = devicePixelRatioF();

    QSize realSize = scaledSize * dpi;
    QPixmap resultPixmap(realSize);

    resultPixmap.fill(Qt::transparent);
    QPainter painter(&resultPixmap);
    painter.setRenderHints(QPainter::Antialiasing | QPainter::SmoothPixmapTransform);

    //画圆
//    QPainterPath path;
//    path.addEllipse(0, 0, realSize.width(), realSize.height());
//    painter.setClipPath(path);


    QPoint drawLeftTop(0, 0);
    QSize drawSize = realSize;

    drawSize.setWidth(realSize.width());
    drawSize.setHeight(realSize.width() * scaledSize.height() / scaledSize.width());
    drawLeftTop.setX(0);
    drawLeftTop.setY((realSize.height() - drawSize.height()) / 2);

    // scaled draw
    painter.drawPixmap(drawLeftTop.x(), drawLeftTop.y(), drawSize.width(), drawSize.height(), pixmap.scaled(drawSize, Qt::IgnoreAspectRatio, Qt::SmoothTransformation));

    resultPixmap.setDevicePixelRatio(dpi);
    return resultPixmap;
}

QPixmap ThemeIconLabel::pixmapToRound(const QPixmap &pixmap, const QSize &scaledSize)
{
    qreal dpi = devicePixelRatioF();

    QSize realSize = scaledSize * dpi;
    QPixmap resultPixmap(realSize);

    resultPixmap.fill(Qt::transparent);
    QPainter painter(&resultPixmap);
    painter.setRenderHints(QPainter::Antialiasing | QPainter::SmoothPixmapTransform);

    QPoint drawLeftTop(0, 0);
    QSize drawSize = realSize;

    drawSize.setWidth(realSize.width());
    drawSize.setHeight(realSize.width() * scaledSize.height() / scaledSize.width());
    drawLeftTop.setX(0);
    drawLeftTop.setY((realSize.height() - drawSize.height()) / 2);

    // scaled draw
    painter.drawPixmap(drawLeftTop.x(), drawLeftTop.y(), drawSize.width(), drawSize.height(), pixmap.scaled(drawSize, Qt::IgnoreAspectRatio, Qt::SmoothTransformation));

    resultPixmap.setDevicePixelRatio(dpi);
    return resultPixmap;
}

void ThemeIconLabel::setRadius(int radius)
{
    mRadius = radius - 2;
}

void ThemeIconLabel::setIcon(const QPixmap &pixmap)
{
    if (mPictrueLabel)
        mPictrueLabel->setIcon(pixmap);
}

void ThemeIconLabel::leaveEvent(QEvent *event)
{
    Q_EMIT leaveWidget();
    QLabel::leaveEvent(event);
}

void ThemeIconLabel::enterEvent(QEvent *event)
{
    Q_EMIT enterWidget();
    QLabel::enterEvent(event);
}
